"""
This module provides a clean interface to all views in the users app.
It uses lazy loading to prevent circular imports.
"""

from importlib import import_module

# List of all view modules and their exports
_VIEW_MODULES = {
    'views': ['LoginView'],  # Add LoginView here
    'user_views': [
        'UserListView',
        'UserDetailView',
        'UserCreateView',
        'UserUpdateView',
        'UserDeleteView',
    ],
    'role_views': [
        'RoleListView',
        'RoleDetailView',
        'RoleCreateView',
        'RoleUpdateView',
        'RoleDeleteView',
    ],
    'department_views': [
        'DepartmentListView',
        'DepartmentDetailView',
        'DepartmentCreateView',
        'DepartmentUpdateView',
        'DepartmentDeleteView',
    ]
}

# Combine all view names for __all__
__all__ = []
for views in _VIEW_MODULES.values():
    __all__.extend(views)

# Lazy loading implementation
class LazyViewLoader:
    def __init__(self):
        self._views = {}
    
    def __getattr__(self, name):
        if name not in self._views:
            for module_name, view_names in _VIEW_MODULES.items():
                if name in view_names:
                    module = import_module(f'.{module_name}', 'users')
                    self._views[name] = getattr(module, name)
                    break
            else:
                raise AttributeError(f"No view named '{name}' found in any view module")
        return self._views[name]

# Create a module-level instance
_views = LazyViewLoader()

def __getattr__(name):
    """Allow direct attribute access to views."""
    try:
        return getattr(_views, name)
    except AttributeError as e:
        raise AttributeError(f"module 'users' has no attribute '{name}'") from e